/**
 * PointWithCharge class is a custom data structure that represents a point in 
 * 3D space with an associated charge. It provides methods to retrieve the 
 * point's coordinates and charge, as well as a string representation of 
 * the point and its charge.
 *
 * @author Christopher Dedman-Rollet
 * @date   11/23/2024
 */

package src;

public class PointWithCharge
{
  private double[] coordinates; // Stores {x, y, z}
  private double charge;        // Stores the charge

  /**
   * Constructs a PointWithCharge with the specified coordinates and charge.
   *
   * @param coordinates an array of three doubles representing the {x, y, z}
   *                    coordinates of the point.
   * @param charge      the charge associated with the point.
   * @throws IllegalArgumentException if the coordinates array is not exactly of
   *                                  length 3.
   */
  public PointWithCharge(double[] coordinates, double charge)
  {
    if (coordinates == null || coordinates.length != 3)
    {
      throw new IllegalArgumentException("Coordinates array must contain exactly three elements (x, y, z).");
    }

    this.coordinates = coordinates;
    this.charge = charge;
  }

  /**
   * Returns the 3D coordinates of the point.
   *
   * @return a double array {x, y, z} representing the coordinates of the point.
   */
  public double[] getCoordinates()
  {
    return coordinates;
  }

  /**
   * Returns the charge associated with the point.
   *
   * @return the charge as a double.
   */
  public double getCharge()
  {
    return charge;
  }

  /**
   * Returns a string representation of the {@code PointWithCharge}.
   *
   * The format is: {{x, y, z}, charge}, where x, y, and z are formatted to three decimal places.
   *
   * @return a string representation of the point and its charge.
   */
  @Override
  public String toString()
  {
    return String.format("{{%.3f, %.3f, %.3f}, %f}", coordinates[0], coordinates[1], coordinates[2], charge);
  }
}
